<?php

namespace App\Traits;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

trait HandlesValidation
{
    public function HandlesValidation(Request $request, array $rules, array $messages = [], ?Closure $after = null, bool $fromApp = false)
    {
        // Sanitize input before validation
        $sanitized = $this->sanitizeInput($request->all());
        $request->merge($sanitized);

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($after) {
            $validator->after($after);
        }

        if ($validator->fails()) {

            if ($fromApp) {
                return response()->json([
                    'error'   => true,
                    'message' => $validator->errors()->first(),
                    'code'    => 102,
                ]);
            }

          throw new \Illuminate\Validation\ValidationException(
                        Validator::make([], []), // empty validator
                        response()->json([
                            'error'   => true,
                            'message' => $validator->errors()->first(),
                            'code'    => 103,
                        ], 422)
                    );
        }

        return null;
    }

    /**
     * Strip tags and dangerous patterns from all request inputs
     */
    protected function sanitizeInput(array $inputs): array
    {
        return array_map(function ($value) {
            if (is_array($value)) {
                return $this->sanitizeInput($value);
            }
            if (is_string($value)) {
                // Detect blacklisted tags/patterns
                if (preg_match('/<(script|iframe|meta|object|embed|link|style|base|form|input|button|textarea|svg|math)/i', $value)) {
                    throw new \Illuminate\Validation\ValidationException(
                        Validator::make([], []), // empty validator
                        response()->json([
                            'error'   => true,
                            'message' => 'Input contains disallowed HTML/JS.',
                            'code'    => 103,
                        ], 422)
                    );
                }

                if (preg_match('/javascript:/i', $value) || preg_match('/on\w+=/i', $value)) {
                    throw new \Illuminate\Validation\ValidationException(
                        Validator::make([], []),
                        response()->json([
                            'error'   => true,
                            'message' => 'Input contains potentially unsafe content.',
                            'code'    => 104,
                        ], 422)
                    );
                }

                // If clean → strip any leftover tags
                return strip_tags(trim($value));
            }
            return $value;
        }, $inputs);
    }
}
